import React from 'react';
import fetch from 'isomorphic-unfetch';
const API = process.env.API;

console.log('🏗️ Notes component has been imported');

const debounce = (fn, delay) => {
  let timer = null;
  return (...args) => {
    clearTimeout(timer);
    timer = setTimeout(() => fn.apply(this, args), delay);
  };
};

// Yes, I mixed React class state with hooks (from logic)
// https://media.giphy.com/media/134qZfT6JZrJsI/giphy.gif
export default class Note extends React.Component {
  state = { note: null, loading: true };

  async componentDidMount() {
    const { slug, token } = this.props;
    const res = await fetch(`${API}/user/notes/${slug}`, {
      headers: {
        authorization: `bearer ${token}`,
      },
    });
    const json = await res.json();

    this.setState({ note: json, loading: false });

    this.saveNote = debounce(contents => {
      fetch(`${API}/user/notes/${slug}`, {
        method: 'post',
        headers: {
          authorization: `bearer ${token}`,
          'content-type': 'application/json',
        },
        body: JSON.stringify({ contents }),
      });
    }, 200);
  }

  handleChange = event => this.saveNote(event.target.value);

  render() {
    const { note, loading } = this.state;

    if (loading) {
      return <p>Loading notes…</p>;
    }

    return (
      <div>
        <style jsx>{`
          label {
            font-weight: bold;
            margin-bottom: 10px;
          }
          textarea {
            padding: 5px;
            display: block;
            border: 1px solid #ccc;
            border-radius: 2px;
            width: 100%;
            margin: 10px 0 40px 0;
          }
        `}</style>
        <label htmlFor="note">Your personal notes</label>
        <textarea
          onChange={this.handleChange}
          rows="4"
          id="note"
          defaultValue={note.contents}
        />
      </div>
    );
  }
}
